<?php

require_once __DIR__ . '/BaseModel.php';

/**
 * مدل مقالات خبری
 * News Articles Model
 */

class Article extends BaseModel
{
    protected string $table = 'articles';
    protected array $fillable = [
        'hash_id', 'title', 'description', 'content', 'url', 'image_url',
        'source', 'published_at', 'category', 'language', 'translated_title',
        'translated_description', 'translated_content', 'posted_at', 'post_id',
        'views_count', 'likes_count', 'shares_count', 'is_featured', 'priority',
        'tags', 'metadata'
    ];

    /**
     * پیدا کردن مقاله بر اساس هش
     */
    public function findByHash(string $hashId): ?array
    {
        return $this->findBy('hash_id', $hashId);
    }

    /**
     * دریافت مقالات ارسال شده
     */
    public function getPosted(int $limit = 20, int $offset = 0): array
    {
        $sql = "SELECT * FROM {$this->table} 
                WHERE posted_at IS NOT NULL 
                ORDER BY posted_at DESC 
                LIMIT :limit OFFSET :offset";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute(['limit' => $limit, 'offset' => $offset]);
        
        $results = $stmt->fetchAll();
        return array_map([$this, 'hideFields'], $results);
    }

    /**
     * دریافت مقالات بر اساس دسته‌بندی
     */
    public function getByCategory(string $category, int $limit = 20): array
    {
        return $this->all(['category' => $category], $limit);
    }

    /**
     * دریافت مقالات بر اساس منبع
     */
    public function getBySource(string $source, int $limit = 20): array
    {
        return $this->all(['source' => $source], $limit);
    }

    /**
     * دریافت مقالات بر اساس زبان
     */
    public function getByLanguage(string $language, int $limit = 20): array
    {
        return $this->all(['language' => $language], $limit);
    }

    /**
     * دریافت مقالات مهم
     */
    public function getFeatured(int $limit = 10): array
    {
        $sql = "SELECT * FROM {$this->table} 
                WHERE is_featured = 1 
                ORDER BY priority DESC, published_at DESC 
                LIMIT :limit";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute(['limit' => $limit]);
        
        $results = $stmt->fetchAll();
        return array_map([$this, 'hideFields'], $results);
    }

    /**
     * جستجو در مقالات
     */
    public function search(string $query, int $limit = 20): array
    {
        $sql = "SELECT * FROM {$this->table} 
                WHERE title LIKE :query 
                   OR description LIKE :query 
                   OR content LIKE :query
                   OR translated_title LIKE :query
                   OR translated_description LIKE :query
                ORDER BY published_at DESC 
                LIMIT :limit";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([
            'query' => "%$query%",
            'limit' => $limit
        ]);
        
        $results = $stmt->fetchAll();
        return array_map([$this, 'hideFields'], $results);
    }

    /**
     * دریافت آمار مقالات
     */
    public function getStats(): array
    {
        $stats = [];
        
        // تعداد کل مقالات
        $stmt = $this->pdo->query("SELECT COUNT(*) FROM {$this->table}");
        $stats['total'] = $stmt->fetchColumn();
        
        // تعداد مقالات ارسال شده
        $stmt = $this->pdo->query("SELECT COUNT(*) FROM {$this->table} WHERE posted_at IS NOT NULL");
        $stats['posted'] = $stmt->fetchColumn();
        
        // تعداد مقالات امروز
        $stmt = $this->pdo->query("SELECT COUNT(*) FROM {$this->table} WHERE DATE(created_at) = DATE('now')");
        $stats['today'] = $stmt->fetchColumn();
        
        // تعداد مقالات این هفته
        $stmt = $this->pdo->query("SELECT COUNT(*) FROM {$this->table} WHERE created_at >= DATE('now', '-7 days')");
        $stats['this_week'] = $stmt->fetchColumn();
        
        // تعداد مقالات این ماه
        $stmt = $this->pdo->query("SELECT COUNT(*) FROM {$this->table} WHERE created_at >= DATE('now', 'start of month')");
        $stats['this_month'] = $stmt->fetchColumn();
        
        // آمار بر اساس دسته‌بندی
        $stmt = $this->pdo->query("SELECT category, COUNT(*) as count FROM {$this->table} GROUP BY category");
        $stats['by_category'] = $stmt->fetchAll();
        
        // آمار بر اساس منبع
        $stmt = $this->pdo->query("SELECT source, COUNT(*) as count FROM {$this->table} GROUP BY source ORDER BY count DESC LIMIT 10");
        $stats['by_source'] = $stmt->fetchAll();
        
        // آمار بر اساس زبان
        $stmt = $this->pdo->query("SELECT language, COUNT(*) as count FROM {$this->table} GROUP BY language");
        $stats['by_language'] = $stmt->fetchAll();
        
        return $stats;
    }

    /**
     * دریافت مقالات محبوب
     */
    public function getPopular(int $limit = 10): array
    {
        $sql = "SELECT * FROM {$this->table} 
                WHERE posted_at IS NOT NULL 
                ORDER BY (views_count + likes_count + shares_count) DESC 
                LIMIT :limit";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute(['limit' => $limit]);
        
        $results = $stmt->fetchAll();
        return array_map([$this, 'hideFields'], $results);
    }

    /**
     * دریافت مقالات اخیر
     */
    public function getRecent(int $limit = 10): array
    {
        $sql = "SELECT * FROM {$this->table} 
                ORDER BY published_at DESC 
                LIMIT :limit";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute(['limit' => $limit]);
        
        $results = $stmt->fetchAll();
        return array_map([$this, 'hideFields'], $results);
    }

    /**
     * دریافت مقالات ترجمه نشده
     */
    public function getUntranslated(int $limit = 20): array
    {
        $sql = "SELECT * FROM {$this->table} 
                WHERE language = 'en' 
                  AND (translated_title IS NULL OR translated_title = '')
                ORDER BY published_at DESC 
                LIMIT :limit";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute(['limit' => $limit]);
        
        $results = $stmt->fetchAll();
        return array_map([$this, 'hideFields'], $results);
    }

    /**
     * به‌روزرسانی آمار بازدید
     */
    public function incrementViews(int $id): bool
    {
        $sql = "UPDATE {$this->table} SET views_count = views_count + 1 WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute(['id' => $id]);
    }

    /**
     * به‌روزرسانی آمار لایک
     */
    public function incrementLikes(int $id): bool
    {
        $sql = "UPDATE {$this->table} SET likes_count = likes_count + 1 WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute(['id' => $id]);
    }

    /**
     * به‌روزرسانی آمار اشتراک
     */
    public function incrementShares(int $id): bool
    {
        $sql = "UPDATE {$this->table} SET shares_count = shares_count + 1 WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute(['id' => $id]);
    }

    /**
     * علامت‌گذاری به عنوان ویژه
     */
    public function markAsFeatured(int $id): bool
    {
        return $this->update($id, ['is_featured' => 1]);
    }

    /**
     * حذف علامت ویژه
     */
    public function unmarkAsFeatured(int $id): bool
    {
        return $this->update($id, ['is_featured' => 0]);
    }

    /**
     * تنظیم اولویت
     */
    public function setPriority(int $id, int $priority): bool
    {
        return $this->update($id, ['priority' => $priority]);
    }

    /**
     * پاک‌سازی مقالات قدیمی
     */
    public function cleanup(int $daysOld = 30): int
    {
        $cutoffDate = date('Y-m-d H:i:s', strtotime("-$daysOld days"));
        
        $sql = "DELETE FROM {$this->table} WHERE created_at < :cutoff_date AND posted_at IS NULL";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute(['cutoff_date' => $cutoffDate]);
        
        return $stmt->rowCount();
    }

    /**
     * دریافت مقالات بر اساس تاریخ
     */
    public function getByDateRange(string $startDate, string $endDate, int $limit = 50): array
    {
        $sql = "SELECT * FROM {$this->table} 
                WHERE published_at BETWEEN :start_date AND :end_date 
                ORDER BY published_at DESC 
                LIMIT :limit";
        
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([
            'start_date' => $startDate,
            'end_date' => $endDate,
            'limit' => $limit
        ]);
        
        $results = $stmt->fetchAll();
        return array_map([$this, 'hideFields'], $results);
    }

    /**
     * دریافت مقالات بر اساس تگ‌ها
     */
    public function getByTags(array $tags, int $limit = 20): array
    {
        $tagConditions = [];
        $params = [];
        
        foreach ($tags as $index => $tag) {
            $tagConditions[] = "tags LIKE :tag_$index";
            $params["tag_$index"] = "%$tag%";
        }
        
        $sql = "SELECT * FROM {$this->table} 
                WHERE " . implode(' OR ', $tagConditions) . "
                ORDER BY published_at DESC 
                LIMIT :limit";
        
        $params['limit'] = $limit;
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute($params);
        
        $results = $stmt->fetchAll();
        return array_map([$this, 'hideFields'], $results);
    }
}
