<?php

/**
 * کلاس ربات تلگرام پیشرفته
 * Advanced Telegram Bot with Management Features
 */

use GuzzleHttp\Client;
use Monolog\Logger;
use Monolog\Handler\StreamHandler;

class TelegramBot
{
    private Client $client;
    private string $botToken;
    private string $channelId;
    private string $baseUrl;
    private DatabaseManager $db;
    private TranslationService $translator;
    private NewsFetcher $newsFetcher;
    private Logger $logger;
    private array $stats = [];
    private int $lastUpdateId = 0;

    public function __construct(string $botToken, string $channelId, DatabaseManager $db)
    {
        $this->botToken = $botToken;
        $this->channelId = $channelId;
        $this->baseUrl = "https://api.cwrtex.ir/bot{$botToken}";
        $this->db = $db;
        
        $this->logger = new Logger('TelegramBot');
        $this->logger->pushHandler(new StreamHandler(__DIR__ . '/../storage/logs/telegram_bot.log'));
        
        $this->client = new Client([
            'timeout' => 30,
            'connect_timeout' => 10,
            'verify' => false, // برای حل مشکل SSL
            'headers' => [
                'User-Agent' => 'AdvancedNewsBot/1.0',
                'Connection' => 'keep-alive'
            ],
            'curl' => [
                CURLOPT_TIMEOUT => 30,
                CURLOPT_CONNECTTIMEOUT => 10,
                CURLOPT_FOLLOWLOCATION => true,
                CURLOPT_MAXREDIRS => 3,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSL_VERIFYHOST => false,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_USERAGENT => 'AdvancedNewsBot/1.0'
            ]
        ]);
        
        $this->translator = new TranslationService($db);
        $this->newsFetcher = new NewsFetcher($db);
        
        $this->loadStats();
    }

    /**
     * ارسال پیام متنی
     */
    public function sendMessage(string $text, string $parseMode = "HTML", array $options = []): bool
    {
        try {
            $payload = array_merge([
                'chat_id' => $this->channelId,
                'text' => $text,
                'parse_mode' => $parseMode
            ], $options);
            
            $response = $this->client->post("{$this->baseUrl}/sendMessage", [
                'form_params' => $payload,
                'timeout' => 15
            ]);
            
            $result = json_decode($response->getBody(), true);
            
            if ($result['ok']) {
                $this->logger->info("پیام متنی ارسال شد");
                $this->updateStats('message_sent');
                return true;
            } else {
                $this->logger->error("خطا در ارسال پیام: " . ($result['description'] ?? 'خطای نامشخص'));
                $this->updateStats('error');
                return false;
            }
            
        } catch (RequestException $e) {
            $this->logger->error("خطا در ارسال پیام: " . $e->getMessage());
            $this->updateStats('error');
            return false;
        }
    }

    /**
     * ارسال عکس با کپشن
     */
    public function sendPhotoWithCaption(string $photoUrl, string $caption, string $parseMode = "HTML", array $options = []): bool
    {
        if (empty($photoUrl) || !filter_var($photoUrl, FILTER_VALIDATE_URL)) {
            return $this->sendMessage($caption, $parseMode, $options);
        }
        
        try {
            $payload = array_merge([
                'chat_id' => $this->channelId,
                'photo' => $photoUrl,
                'caption' => $caption,
                'parse_mode' => $parseMode
            ], $options);
            
            $response = $this->client->post("{$this->baseUrl}/sendPhoto", [
                'form_params' => $payload,
                'timeout' => 20
            ]);
            
            $result = json_decode($response->getBody(), true);
            
            if ($result['ok']) {
                $this->logger->info("عکس با کپشن ارسال شد");
                $this->updateStats('photo_sent');
                return true;
            } else {
                $this->logger->error("خطا در ارسال عکس: " . ($result['description'] ?? 'خطای نامشخص'));
                // تلاش برای ارسال متن به جای عکس
                return $this->sendMessage($caption, $parseMode, $options);
            }
            
        } catch (RequestException $e) {
            $this->logger->error("خطا در ارسال عکس: " . $e->getMessage());
            // تلاش برای ارسال متن به جای عکس
            return $this->sendMessage($caption, $parseMode, $options);
        }
    }

    /**
     * ارسال مقاله خبری
     */
    public function sendNewsArticle(array $article): bool
    {
        try {
            // ترجمه محتوا
            if (empty($article['translated_title'])) {
                $article['translated_title'] = $this->translator->translateText($article['title']);
            }
            
            if (empty($article['translated_description'])) {
                $article['translated_description'] = $this->translator->translateText($article['description']);
            }
            
            // ایجاد کپشن
            $caption = $this->createNewsCaption($article);
            
            // ارسال
            $success = $this->sendPhotoWithCaption($article['image_url'], $caption);
            
            if ($success) {
                // ذخیره در پایگاه داده
                $this->saveArticleToDatabase($article);
                $this->logger->info("مقاله خبری ارسال شد: " . $article['title']);
            }
            
            return $success;
            
        } catch (Exception $e) {
            $this->logger->error("خطا در ارسال مقاله: " . $e->getMessage());
            $this->updateStats('error');
            return false;
        }
    }

    /**
     * ایجاد کپشن برای مقاله
     */
    private function createNewsCaption(array $article): string
    {
        // ایموجی بر اساس دسته‌بندی
        $emojiMap = [
            'crypto' => '₿',
            'forex' => '💱',
            'iran' => '🇮🇷',
            'economy' => '📈',
            'politics' => '🏛️',
            'technology' => '💻',
            'general' => '📰'
        ];
        
        $emoji = $emojiMap[$article['category']] ?? '📰';
        
        // ایجاد کپشن
        $caption = "{$emoji} <b>" . htmlspecialchars($article['translated_title']) . "</b>\n\n";
        
        if (!empty($article['translated_description'])) {
            $description = htmlspecialchars($article['translated_description']);
            // محدود کردن طول توضیحات
            if (strlen($description) > 200) {
                $description = substr($description, 0, 200) . '...';
            }
            $caption .= "{$description}\n\n";
        }
        
        // اطلاعات منبع و زمان
        $publishedAt = new DateTime($article['published_at']);
        $caption .= "📅 " . $this->formatPersianDate($publishedAt) . "\n";
        $caption .= "📰 منبع: " . htmlspecialchars($article['source']) . "\n";
        $caption .= "🔗 <a href='" . htmlspecialchars($article['url']) . "'>مطالعه بیشتر</a>";
        
        // اضافه کردن تگ‌ها
        if (!empty($article['tags'])) {
            $tags = is_array($article['tags']) ? implode(' ', $article['tags']) : $article['tags'];
            $caption .= "\n\n#{$tags}";
        }
        
        return $caption;
    }

    /**
     * فرمت تاریخ به فارسی
     */
    private function formatPersianDate(DateTime $date): string
    {
        // اینجا می‌توانید از کتابخانه تاریخ فارسی استفاده کنید
        // فعلاً فرمت ساده استفاده می‌کنیم
        return $date->format('Y/m/d H:i');
    }

    /**
     * ذخیره مقاله در پایگاه داده
     */
    private function saveArticleToDatabase(array $article): void
    {
        try {
            $article['posted_at'] = date('Y-m-d H:i:s');
            
            $sql = "INSERT OR REPLACE INTO articles 
                    (hash_id, title, description, content, url, image_url, source, 
                     published_at, category, language, translated_title, 
                     translated_description, translated_content, posted_at) 
                    VALUES (:hash_id, :title, :description, :content, :url, :image_url, :source, 
                            :published_at, :category, :language, :translated_title, 
                            :translated_description, :translated_content, :posted_at)";
            
            $this->db->query($sql, $article);
            
        } catch (Exception $e) {
            $this->logger->error("خطا در ذخیره مقاله در پایگاه داده: " . $e->getMessage());
        }
    }

    /**
     * بررسی امکان ارسال پست
     */
    public function canPostNow(): bool
    {
        $rateLimits = Config::get('rate_limits', []);
        
        // بررسی محدودیت ساعتی
        $hourlyLimit = $rateLimits['max_posts_per_hour'] ?? 10;
        if ($this->stats['posts_this_hour'] >= $hourlyLimit) {
            return false;
        }
        
        // بررسی محدودیت روزانه
        $dailyLimit = $rateLimits['max_posts_per_day'] ?? 50;
        if ($this->stats['posts_today'] >= $dailyLimit) {
            return false;
        }
        
        // بررسی فاصله زمانی بین پست‌ها
        $minInterval = $rateLimits['min_news_interval'] ?? 5;
        if (!empty($this->stats['last_post_time'])) {
            $lastPostTime = new DateTime($this->stats['last_post_time']);
            $now = new DateTime();
            $diff = $now->getTimestamp() - $lastPostTime->getTimestamp();
            
            if ($diff < $minInterval * 60) {
                return false;
            }
        }
        
        return true;
    }

    /**
     * دریافت پیام‌های جدید
     */
    public function getUpdates(): array
    {
        try {
            $response = $this->client->get("{$this->baseUrl}/getUpdates", [
                'query' => [
                    'offset' => $this->lastUpdateId + 1,
                    'timeout' => 10
                ],
                'timeout' => 15
            ]);
            
            $result = json_decode($response->getBody(), true);
            
            if ($result['ok']) {
                $updates = $result['result'];
                if (!empty($updates)) {
                    $this->lastUpdateId = end($updates)['update_id'];
                }
                return $updates;
            }
            
        } catch (RequestException $e) {
            $this->logger->error("خطا در دریافت پیام‌ها: " . $e->getMessage());
        }
        
        return [];
    }

    /**
     * ارسال پیام خصوصی
     */
    public function sendPrivateMessage(int $chatId, string $text, string $parseMode = "HTML"): bool
    {
        try {
            $response = $this->client->post("{$this->baseUrl}/sendMessage", [
                'form_params' => [
                    'chat_id' => $chatId,
                    'text' => $text,
                    'parse_mode' => $parseMode
                ],
                'timeout' => 30
            ]);
            
            $result = json_decode($response->getBody(), true);
            
            if ($result['ok']) {
                $this->logger->info("پیام خصوصی ارسال شد به $chatId");
                return true;
            } else {
                $this->logger->error("خطا در ارسال پیام خصوصی: " . ($result['description'] ?? 'خطای نامشخص'));
                return false;
            }
            
        } catch (RequestException $e) {
            $this->logger->error("خطا در ارسال پیام خصوصی: " . $e->getMessage());
            return false;
        }
    }

    /**
     * پردازش پیام خصوصی
     */
    public function handlePrivateMessage(array $message): void
    {
        try {
            $chatId = $message['chat']['id'];
            $text = $message['text'] ?? '';
            $userId = $message['from']['id'];
            $username = $message['from']['username'] ?? '';
            
            $this->logger->info("پیام از {$username} ({$userId}): " . substr($text, 0, 50) . "...");
            
            // بررسی دسترسی ادمین
            if (!$this->isAdmin($userId)) {
                $this->sendPrivateMessage($chatId, "❌ شما دسترسی ادمین ندارید.");
                return;
            }
            
            // دستورات ربات
            if (strpos($text, '/') === 0) {
                $this->handleCommand($chatId, $text, $userId, $username);
            } else {
                $this->sendHelpMessage($chatId);
            }
            
        } catch (Exception $e) {
            $this->logger->error("خطا در پردازش پیام خصوصی: " . $e->getMessage());
        }
    }

    /**
     * پردازش دستورات
     */
    private function handleCommand(int $chatId, string $command, int $userId, string $username): void
    {
        $command = trim($command);
        
        switch ($command) {
            case '/start':
                $this->sendWelcomeMessage($chatId);
                break;
                
            case '/help':
                $this->sendHelpMessage($chatId);
                break;
                
            case '/status':
                $this->sendStatusMessage($chatId);
                break;
                
            case '/stats':
                $this->sendStatsMessage($chatId);
                break;
                
            case '/news':
                $this->sendLatestNews($chatId);
                break;
                
            case '/test':
                $this->sendTestMessage($chatId);
                break;
                
            case '/fetch':
                $this->fetchAndSendNews($chatId);
                break;
                
            case '/settings':
                $this->sendSettingsMessage($chatId);
                break;
                
            default:
                $this->sendPrivateMessage($chatId, "❓ دستور نامشخص!\n\nبرای دریافت راهنما /help را ارسال کنید.");
        }
    }

    /**
     * ارسال پیام خوش‌آمدگویی
     */
    private function sendWelcomeMessage(int $chatId): void
    {
        $message = "🤖 <b>خوش آمدید!</b>\n\n";
        $message .= "من ربات اخبار پیشرفته هستم که اخبار مهم جهان و ایران را دریافت و ترجمه می‌کنم.\n\n";
        $message .= "📰 <b>اخبار شامل:</b>\n";
        $message .= "• اخبار اقتصادی و فارکس\n";
        $message .= "• اخبار کریپتو و بیت‌کوین\n";
        $message .= "• اخبار سیاسی ایران و جهان\n";
        $message .= "• اخبار فناوری\n\n";
        $message .= "برای دریافت راهنما /help را ارسال کنید.";
        
        $this->sendPrivateMessage($chatId, $message);
    }

    /**
     * ارسال پیام راهنما
     */
    private function sendHelpMessage(int $chatId): void
    {
        $message = "📋 <b>راهنمای ربات اخبار:</b>\n\n";
        $message .= "🔹 /start - شروع ربات\n";
        $message .= "🔹 /help - نمایش این راهنما\n";
        $message .= "🔹 /status - وضعیت فعلی ربات\n";
        $message .= "🔹 /stats - آمار کامل ربات\n";
        $message .= "🔹 /news - آخرین اخبار دریافت شده\n";
        $message .= "🔹 /test - تست اتصال\n";
        $message .= "🔹 /fetch - دریافت و ارسال اخبار جدید\n";
        $message .= "🔹 /settings - تنظیمات ربات\n\n";
        $message .= "📱 ربات به صورت خودکار اخبار مهم را در کانال ارسال می‌کند.";
        
        $this->sendPrivateMessage($chatId, $message);
    }

    /**
     * ارسال وضعیت ربات
     */
    private function sendStatusMessage(int $chatId): void
    {
        $message = "📊 <b>وضعیت ربات:</b>\n\n";
        $message .= "🟢 وضعیت: " . ($this->canPostNow() ? "فعال" : "غیرفعال") . "\n";
        $message .= "📈 کل پست‌ها: " . $this->stats['total_posts'] . "\n";
        $message .= "📅 پست‌های امروز: " . $this->stats['posts_today'] . "\n";
        $message .= "⏰ آخرین پست: " . ($this->stats['last_post_time'] ?: 'هنوز پستی ارسال نشده') . "\n";
        $message .= "❌ خطاها: " . $this->stats['errors_count'] . "\n";
        $message .= "✅ امکان ارسال: " . ($this->canPostNow() ? "بله" : "خیر");
        
        $this->sendPrivateMessage($chatId, $message);
    }

    /**
     * ارسال آمار کامل
     */
    private function sendStatsMessage(int $chatId): void
    {
        $message = "📊 <b>آمار کامل ربات:</b>\n\n";
        $message .= "📈 کل پست‌ها: " . $this->stats['total_posts'] . "\n";
        $message .= "📅 پست‌های امروز: " . $this->stats['posts_today'] . "\n";
        $message .= "⏰ پست‌های این ساعت: " . $this->stats['posts_this_hour'] . "\n";
        $message .= "📊 پست‌های این هفته: " . $this->stats['posts_this_week'] . "\n";
        $message .= "📊 پست‌های این ماه: " . $this->stats['posts_this_month'] . "\n";
        $message .= "🕐 آخرین پست: " . ($this->stats['last_post_time'] ?: 'هنوز پستی ارسال نشده') . "\n";
        $message .= "❌ تعداد خطاها: " . $this->stats['errors_count'] . "\n";
        $message .= "📰 مقالات دریافت شده: " . $this->stats['total_articles_fetched'] . "\n";
        $message .= "🌐 ترجمه‌ها: " . $this->stats['total_translations'] . "\n";
        
        if (!empty($this->stats['last_error'])) {
            $message .= "\n⚠️ آخرین خطا: " . substr($this->stats['last_error'], 0, 100) . "...";
        }
        
        $this->sendPrivateMessage($chatId, $message);
    }

    /**
     * ارسال آخرین اخبار
     */
    private function sendLatestNews(int $chatId): void
    {
        try {
            $sql = "SELECT * FROM articles WHERE posted_at IS NOT NULL ORDER BY posted_at DESC LIMIT 5";
            $articles = $this->db->query($sql);
            
            if (empty($articles)) {
                $this->sendPrivateMessage($chatId, "📰 هنوز هیچ خبری ارسال نشده است.");
                return;
            }
            
            $message = "📰 <b>آخرین اخبار ارسال شده:</b>\n\n";
            
            foreach ($articles as $i => $article) {
                $message .= ($i + 1) . ". " . ($article['translated_title'] ?: $article['title']) . "\n";
                $message .= "   📅 " . date('Y/m/d H:i', strtotime($article['published_at'])) . "\n";
                $message .= "   📰 " . $article['source'] . "\n\n";
            }
            
            $this->sendPrivateMessage($chatId, $message);
            
        } catch (Exception $e) {
            $this->logger->error("خطا در دریافت آخرین اخبار: " . $e->getMessage());
            $this->sendPrivateMessage($chatId, "❌ خطا در دریافت اخبار.");
        }
    }

    /**
     * ارسال پیام تست
     */
    private function sendTestMessage(int $chatId): void
    {
        $testMessage = "🧪 <b>تست اتصال ربات</b>\n\n";
        $testMessage .= "✅ اتصال به تلگرام: موفق\n";
        $testMessage .= "✅ اتصال به پایگاه داده: موفق\n";
        $testMessage .= "✅ سرویس ترجمه: فعال\n";
        $testMessage .= "✅ دریافت‌کننده اخبار: فعال\n\n";
        $testMessage .= "🕐 زمان تست: " . date('Y/m/d H:i:s');
        
        $this->sendPrivateMessage($chatId, $testMessage);
    }

    /**
     * دریافت و ارسال اخبار جدید
     */
    private function fetchAndSendNews(int $chatId): void
    {
        $this->sendPrivateMessage($chatId, "🔄 در حال دریافت اخبار جدید...");
        
        try {
            $articles = $this->newsFetcher->fetchAllNews();
            
            if (empty($articles)) {
                $this->sendPrivateMessage($chatId, "❌ هیچ خبر جدیدی یافت نشد.");
                return;
            }
            
            $importantArticles = $this->filterImportantNews($articles);
            
            if (empty($importantArticles)) {
                $this->sendPrivateMessage($chatId, "❌ هیچ خبر مهمی یافت نشد.");
                return;
            }
            
            $postedCount = 0;
            foreach (array_slice($importantArticles, 0, 3) as $article) {
                if ($this->sendNewsArticle($article)) {
                    $postedCount++;
                    sleep(2); // فاصله بین پست‌ها
                }
            }
            
            $this->sendPrivateMessage($chatId, "✅ {$postedCount} خبر جدید ارسال شد.");
            
        } catch (Exception $e) {
            $this->logger->error("خطا در دریافت اخبار: " . $e->getMessage());
            $this->sendPrivateMessage($chatId, "❌ خطا در دریافت اخبار.");
        }
    }

    /**
     * فیلتر کردن اخبار مهم
     */
    private function filterImportantNews(array $articles): array
    {
        $important = [];
        $keywords = Config::get('keywords', []);
        $allKeywords = array_merge(...array_values($keywords));
        
        foreach ($articles as $article) {
            $text = strtolower($article['title'] . ' ' . $article['description']);
            
            foreach ($allKeywords as $keyword) {
                if (strpos($text, $keyword) !== false) {
                    $important[] = $article;
                    break;
                }
            }
        }
        
        // مرتب‌سازی بر اساس تاریخ انتشار
        usort($important, function($a, $b) {
            return strtotime($b['published_at']) - strtotime($a['published_at']);
        });
        
        return array_slice($important, 0, 10);
    }

    /**
     * ارسال پیام تنظیمات
     */
    private function sendSettingsMessage(int $chatId): void
    {
        $message = "⚙️ <b>تنظیمات ربات:</b>\n\n";
        $message .= "📊 حداکثر پست در ساعت: " . (Config::get('rate_limits.max_posts_per_hour', 10)) . "\n";
        $message .= "📊 حداکثر پست در روز: " . (Config::get('rate_limits.max_posts_per_day', 50)) . "\n";
        $message .= "⏰ حداقل فاصله بین اخبار: " . (Config::get('rate_limits.min_news_interval', 5)) . " دقیقه\n";
        $message .= "🌐 سرویس ترجمه: " . (Config::get('apis.translation.service', 'mymemory')) . "\n";
        $message .= "🔄 فاصله بررسی اخبار: " . (Config::get('scheduler.news_check_interval', 30)) . " دقیقه\n";
        
        $this->sendPrivateMessage($chatId, $message);
    }

    /**
     * بررسی دسترسی ادمین
     */
    private function isAdmin(int $userId): bool
    {
        try {
            $sql = "SELECT COUNT(*) FROM admin_users WHERE user_id = :user_id AND is_active = 1";
            $stmt = $this->db->query($sql, ['user_id' => $userId]);
            return $stmt->fetchColumn() > 0;
        } catch (Exception $e) {
            $this->logger->error("خطا در بررسی دسترسی ادمین: " . $e->getMessage());
            return false;
        }
    }

    /**
     * بارگذاری آمار
     */
    private function loadStats(): void
    {
        try {
            $sql = "SELECT * FROM bot_stats WHERE id = 1";
            $stmt = $this->db->query($sql);
            $this->stats = $stmt->fetch() ?: [];
        } catch (Exception $e) {
            $this->logger->error("خطا در بارگذاری آمار: " . $e->getMessage());
            $this->stats = [];
        }
    }

    /**
     * به‌روزرسانی آمار
     */
    private function updateStats(string $type): void
    {
        try {
            $now = date('Y-m-d H:i:s');
            
            switch ($type) {
                case 'message_sent':
                case 'photo_sent':
                    $this->stats['total_posts'] = ($this->stats['total_posts'] ?? 0) + 1;
                    $this->stats['posts_today'] = ($this->stats['posts_today'] ?? 0) + 1;
                    $this->stats['posts_this_hour'] = ($this->stats['posts_this_hour'] ?? 0) + 1;
                    $this->stats['last_post_time'] = $now;
                    break;
                    
                case 'error':
                    $this->stats['errors_count'] = ($this->stats['errors_count'] ?? 0) + 1;
                    break;
            }
            
            $this->stats['last_activity'] = $now;
            
            // ذخیره در پایگاه داده
            $sql = "UPDATE bot_stats SET 
                        total_posts = :total_posts,
                        posts_today = :posts_today,
                        posts_this_hour = :posts_this_hour,
                        last_post_time = :last_post_time,
                        errors_count = :errors_count,
                        last_activity = :last_activity
                    WHERE id = 1";
            
            $this->db->query($sql, $this->stats);
            
        } catch (Exception $e) {
            $this->logger->error("خطا در به‌روزرسانی آمار: " . $e->getMessage());
        }
    }

    /**
     * بازنشانی آمار ساعتی
     */
    public function resetHourlyStats(): void
    {
        $this->stats['posts_this_hour'] = 0;
        $this->updateStats('reset');
    }

    /**
     * بازنشانی آمار روزانه
     */
    public function resetDailyStats(): void
    {
        $this->stats['posts_today'] = 0;
        $this->updateStats('reset');
    }

    /**
     * دریافت اطلاعات ربات
     */
    public function getBotInfo(): array
    {
        try {
            $response = $this->client->get("{$this->baseUrl}/getMe");
            $result = json_decode($response->getBody(), true);
            
            if ($result['ok']) {
                return $result['result'];
            }
            
        } catch (RequestException $e) {
            $this->logger->error("خطا در دریافت اطلاعات ربات: " . $e->getMessage());
            
            // در صورت عدم اتصال، اطلاعات پیش‌فرض برگردان
            if (strpos($e->getMessage(), 'Failed to connect') !== false) {
                $this->logger->warning("اتصال به تلگرام برقرار نیست - استفاده از حالت آفلاین");
                return [
                    'id' => 123456789,
                    'is_bot' => true,
                    'first_name' => 'Advanced News Bot',
                    'username' => 'advanced_news_bot',
                    'can_join_groups' => false,
                    'can_read_all_group_messages' => false,
                    'supports_inline_queries' => false
                ];
            }
        }
        
        return [];
    }

    /**
     * تنظیم Webhook
     */
    public function setWebhook(string $webhookUrl): bool
    {
        try {
            $response = $this->client->post("{$this->baseUrl}/setWebhook", [
                'form_params' => [
                    'url' => $webhookUrl
                ]
            ]);
            
            $result = json_decode($response->getBody(), true);
            return $result['ok'];
            
        } catch (RequestException $e) {
            $this->logger->error("خطا در تنظیم Webhook: " . $e->getMessage());
            return false;
        }
    }

    /**
     * حذف Webhook
     */
    public function deleteWebhook(): bool
    {
        try {
            $response = $this->client->post("{$this->baseUrl}/deleteWebhook");
            $result = json_decode($response->getBody(), true);
            return $result['ok'];
            
        } catch (RequestException $e) {
            $this->logger->error("خطا در حذف Webhook: " . $e->getMessage());
            return false;
        }
    }
}
