<?php

/**
 * کلاس مدیریت تنظیمات برنامه
 * Configuration Manager Class
 */

class Config
{
    private static array $config = [];
    private static bool $loaded = false;

    /**
     * بارگذاری تنظیمات از فایل
     */
    public static function load(): void
    {
        if (self::$loaded) {
            return;
        }

        $configFile = __DIR__ . '/../config/app.php';
        if (file_exists($configFile)) {
            self::$config = require $configFile;
        }

        // بارگذاری متغیرهای محیطی
        self::loadEnvVariables();
        
        self::$loaded = true;
    }

    /**
     * دریافت مقدار تنظیمات
     */
    public static function get(string $key, $default = null)
    {
        self::load();
        
        $keys = explode('.', $key);
        $value = self::$config;
        
        foreach ($keys as $k) {
            if (!isset($value[$k])) {
                return $default;
            }
            $value = $value[$k];
        }
        
        return $value;
    }

    /**
     * تنظیم مقدار
     */
    public static function set(string $key, $value): void
    {
        self::load();
        
        $keys = explode('.', $key);
        $config = &self::$config;
        
        foreach ($keys as $k) {
            if (!isset($config[$k])) {
                $config[$k] = [];
            }
            $config = &$config[$k];
        }
        
        $config = $value;
    }

    /**
     * بررسی وجود کلید
     */
    public static function has(string $key): bool
    {
        self::load();
        
        $keys = explode('.', $key);
        $value = self::$config;
        
        foreach ($keys as $k) {
            if (!isset($value[$k])) {
                return false;
            }
            $value = $value[$k];
        }
        
        return true;
    }

    /**
     * دریافت تمام تنظیمات
     */
    public static function all(): array
    {
        self::load();
        return self::$config;
    }

    /**
     * بارگذاری متغیرهای محیطی از فایل .env
     */
    private static function loadEnvVariables(): void
    {
        $envFile = __DIR__ . '/../.env';
        if (!file_exists($envFile)) {
            return;
        }

        $lines = file($envFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
        foreach ($lines as $line) {
            if (strpos($line, '=') !== false && strpos($line, '#') !== 0) {
                list($key, $value) = explode('=', $line, 2);
                $key = trim($key);
                $value = trim($value, '"\'');
                
                if (!getenv($key)) {
                    putenv("$key=$value");
                }
            }
        }
    }

    /**
     * دریافت متغیر محیطی
     */
    public static function env(string $key, $default = null)
    {
        $value = getenv($key);
        return $value !== false ? $value : $default;
    }

    /**
     * دریافت تنظیمات ربات
     */
    public static function bot(): array
    {
        return self::get('bot', []);
    }

    /**
     * دریافت تنظیمات API ها
     */
    public static function apis(): array
    {
        return self::get('apis', []);
    }

    /**
     * دریافت تنظیمات پایگاه داده
     */
    public static function database(): array
    {
        return self::get('database', []);
    }

    /**
     * دریافت تنظیمات زمان‌بندی
     */
    public static function scheduler(): array
    {
        return self::get('scheduler', []);
    }

    /**
     * دریافت محدودیت‌های ارسال
     */
    public static function rateLimits(): array
    {
        return self::get('rate_limits', []);
    }

    /**
     * دریافت منابع خبری
     */
    public static function newsSources(): array
    {
        return self::get('news_sources', []);
    }

    /**
     * دریافت کلمات کلیدی
     */
    public static function keywords(): array
    {
        return self::get('keywords', []);
    }

    /**
     * دریافت تنظیمات لاگ
     */
    public static function logging(): array
    {
        return self::get('logging', []);
    }

    /**
     * دریافت تنظیمات کش
     */
    public static function cache(): array
    {
        return self::get('cache', []);
    }

    /**
     * دریافت تنظیمات امنیتی
     */
    public static function security(): array
    {
        return self::get('security', []);
    }

    /**
     * دریافت تنظیمات پنل مدیریت
     */
    public static function admin(): array
    {
        return self::get('admin', []);
    }

    /**
     * دریافت تنظیمات اعلان‌ها
     */
    public static function notifications(): array
    {
        return self::get('notifications', []);
    }

    /**
     * دریافت تنظیمات پشتیبان‌گیری
     */
    public static function backup(): array
    {
        return self::get('backup', []);
    }

    /**
     * دریافت تنظیمات مانیتورینگ
     */
    public static function monitoring(): array
    {
        return self::get('monitoring', []);
    }

    /**
     * اعتبارسنجی تنظیمات
     */
    public static function validate(): array
    {
        $errors = [];
        
        // بررسی توکن ربات
        if (empty(self::get('bot.token'))) {
            $errors[] = 'توکن ربات تلگرام تنظیم نشده است';
        }
        
        // بررسی شناسه کانال
        if (empty(self::get('bot.channel_id'))) {
            $errors[] = 'شناسه کانال تلگرام تنظیم نشده است';
        }
        
        // بررسی کلید API اخبار
        if (empty(self::get('apis.news_api.key'))) {
            $errors[] = 'کلید API اخبار تنظیم نشده است';
        }
        
        // بررسی مسیر پایگاه داده
        $dbPath = self::get('database.path');
        if (empty($dbPath)) {
            $errors[] = 'مسیر پایگاه داده تنظیم نشده است';
        }
        
        return $errors;
    }

    /**
     * ذخیره تنظیمات در فایل
     */
    public static function save(): bool
    {
        try {
            $configFile = __DIR__ . '/../config/app.php';
            $content = "<?php\n\n/**\n * تنظیمات اصلی برنامه\n * Advanced Telegram News Bot - Configuration\n */\n\nreturn " . var_export(self::$config, true) . ";\n";
            
            return file_put_contents($configFile, $content) !== false;
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * ریست کردن تنظیمات
     */
    public static function reset(): void
    {
        self::$config = [];
        self::$loaded = false;
    }
}
