<?php
/**
 * اسکریپت نصب برای هاست اشتراکی
 * Shared Hosting Installation Script
 */

echo "🚀 نصب ربات خبری پیشرفته برای هاست اشتراکی\n";
echo "=" . str_repeat("=", 50) . "\n";

try {
    // بررسی وجود فایل .env
    if (!file_exists(__DIR__ . '/.env')) {
        echo "📝 ایجاد فایل تنظیمات...\n";
        if (file_exists(__DIR__ . '/env.shared_hosting')) {
            copy(__DIR__ . '/env.shared_hosting', __DIR__ . '/.env');
            echo "✅ فایل .env از تنظیمات هاست اشتراکی ایجاد شد\n";
        } else {
            copy(__DIR__ . '/env.example', __DIR__ . '/.env');
            echo "✅ فایل .env ایجاد شد\n";
        }
        echo "⚠️ لطفاً فایل .env را ویرایش کنید و تنظیمات را وارد کنید\n";
    }
    
    // ایجاد پوشه‌های مورد نیاز
    $directories = [
        'storage',
        'storage/logs',
        'storage/cache',
        'storage/backups'
    ];
    
    foreach ($directories as $dir) {
        $path = __DIR__ . '/' . $dir;
        if (!is_dir($path)) {
            if (mkdir($path, 0755, true)) {
                echo "📁 پوشه $dir ایجاد شد\n";
            } else {
                echo "❌ خطا در ایجاد پوشه $dir\n";
            }
        }
    }
    
    // تنظیم مجوزها
    echo "🔐 تنظیم مجوزها...\n";
    $permissions = [
        'storage' => 0755,
        'storage/logs' => 0755,
        'storage/cache' => 0755,
        'storage/backups' => 0755,
        '.env' => 0600
    ];
    
    foreach ($permissions as $path => $perm) {
        $fullPath = __DIR__ . '/' . $path;
        if (file_exists($fullPath) || is_dir($fullPath)) {
            if (chmod($fullPath, $perm)) {
                echo "✅ مجوز $path تنظیم شد\n";
            } else {
                echo "⚠️ خطا در تنظیم مجوز $path\n";
            }
        }
    }
    
    // بررسی وجود vendor
    if (!is_dir(__DIR__ . '/vendor')) {
        echo "📦 بررسی وابستگی‌ها...\n";
        if (file_exists(__DIR__ . '/composer.json')) {
            echo "⚠️ پوشه vendor موجود نیست\n";
            echo "💡 راه‌حل‌ها:\n";
            echo "   1. اجرای 'composer install' در سیستم محلی و آپلود vendor\n";
            echo "   2. استفاده از نسخه Python ربات\n";
            echo "   3. تماس با پشتیبانی هاست برای فعال‌سازی Composer\n";
        } else {
            echo "✅ فایل composer.json موجود نیست - احتمالاً نسخه Python\n";
        }
    } else {
        echo "✅ پوشه vendor موجود است\n";
    }
    
    // تست تنظیمات
    echo "⚙️ بررسی تنظیمات...\n";
    
    if (file_exists(__DIR__ . '/.env')) {
        $envContent = file_get_contents(__DIR__ . '/.env');
        
        $requiredSettings = [
            'BOT_TOKEN' => 'توکن ربات تلگرام',
            'CHANNEL_ID' => 'شناسه کانال تلگرام',
            'NEWS_API_KEY' => 'کلید API اخبار'
        ];
        
        $missingSettings = [];
        foreach ($requiredSettings as $setting => $description) {
            if (strpos($envContent, $setting . '=YOUR_') !== false || 
                strpos($envContent, $setting . '=') === false) {
                $missingSettings[] = $description;
            }
        }
        
        if (!empty($missingSettings)) {
            echo "❌ تنظیمات ناقص:\n";
            foreach ($missingSettings as $setting) {
                echo "   - $setting\n";
            }
            echo "\n⚠️ لطفاً فایل .env را ویرایش کنید\n";
        } else {
            echo "✅ تنظیمات اصلی موجود است\n";
        }
    }
    
    // تست اتصالات (اگر تنظیمات موجود است)
    if (file_exists(__DIR__ . '/.env') && empty($missingSettings)) {
        echo "🧪 تست اتصالات...\n";
        
        // بارگذاری تنظیمات
        require_once __DIR__ . '/src/Config.php';
        
        try {
            $botToken = Config::get('bot.token');
            $apiKey = Config::get('apis.news_api.key');
            
            if ($botToken && $apiKey) {
                $client = new GuzzleHttp\Client(['timeout' => 10]);
                
                // تست اتصال به تلگرام
                try {
                    $response = $client->get("https://api.cwrtex.ir/bot{$botToken}/getMe");
                    $result = json_decode($response->getBody(), true);
                    
                    if ($result['ok']) {
                        echo "✅ اتصال به تلگرام موفق - ربات: " . $result['result']['first_name'] . "\n";
                    } else {
                        echo "❌ خطا در اتصال به تلگرام\n";
                    }
                } catch (Exception $e) {
                    echo "❌ خطا در تست تلگرام: " . $e->getMessage() . "\n";
                }
                
                // تست اتصال به NewsAPI
                try {
                    $response = $client->get("https://newsapi.org/v2/sources?apiKey={$apiKey}");
                    $result = json_decode($response->getBody(), true);
                    
                    if ($result['status'] === 'ok') {
                        echo "✅ اتصال به NewsAPI موفق\n";
                    } else {
                        echo "❌ خطا در اتصال به NewsAPI: " . ($result['message'] ?? 'خطای نامشخص') . "\n";
                    }
                } catch (Exception $e) {
                    echo "❌ خطا در تست NewsAPI: " . $e->getMessage() . "\n";
                }
            }
        } catch (Exception $e) {
            echo "⚠️ خطا در تست اتصالات: " . $e->getMessage() . "\n";
        }
    }
    
    // راهنمای Cron Job
    echo "\n📋 راهنمای تنظیم Cron Job:\n";
    echo "=" . str_repeat("=", 30) . "\n";
    
    $currentPath = __DIR__;
    $phpPath = '/usr/bin/php'; // مسیر پیش‌فرض PHP
    
    echo "برای نسخه PHP:\n";
    echo "*/30 * * * * $phpPath $currentPath/shared_hosting_runner.php\n\n";
    
    echo "برای نسخه Python:\n";
    echo "*/30 * * * * /usr/bin/python3 $currentPath/py/advanced_news_bot.py\n\n";
    
    echo "📝 نکات مهم:\n";
    echo "• مسیر PHP ممکن است متفاوت باشد (/usr/local/bin/php)\n";
    echo "• هر 30 دقیقه یکبار اجرا شود\n";
    echo "• نیازی به Webhook نیست\n";
    echo "• ربات از روش Polling استفاده می‌کند\n\n";
    
    // راهنمای تست
    echo "🧪 دستورات تست:\n";
    echo "=" . str_repeat("=", 20) . "\n";
    echo "php shared_hosting_runner.php --test\n";
    echo "php shared_hosting_runner.php --status\n";
    echo "php shared_hosting_runner.php\n\n";
    
    echo "🎉 نصب با موفقیت انجام شد!\n";
    echo "\n📋 مراحل بعدی:\n";
    echo "1. ویرایش فایل .env\n";
    echo "2. تنظیم Cron Job\n";
    echo "3. تست ربات\n";
    echo "4. مانیتورینگ عملکرد\n";
    
} catch (Exception $e) {
    echo "❌ خطا در نصب: " . $e->getMessage() . "\n";
    exit(1);
}
